package edu.iu.sci2.visualization.geomaps.data.scaling;

import edu.iu.sci2.visualization.geomaps.LogStream;


public enum Scaling {
	Linear {
		@Override
		protected double scaleUnchecked(double value) {
			return value;
		}		

		@Override
		public Scaling inverse() {
			return Scaling.Linear;
		}		
	},
	Logarithmic {
		@Override
		protected double scaleUnchecked(double value) {
			return Math.log10(value);
		}

		@Override
		public Scaling inverse() {
			return Scaling.Exponential;
		}
	},
	Exponential {
		@Override
		protected double scaleUnchecked(double value) {
			return Math.pow(10.0, value);
		}

		@Override
		public Scaling inverse() {
			return Scaling.Logarithmic;
		}		
	};
	
	protected abstract double scaleUnchecked(double value);
	public abstract Scaling inverse();
	
	
	public double scale(double value) throws ScalingException {
		double unchecked = scaleUnchecked(value);
		
		if (Double.isInfinite(unchecked) || Double.isNaN(unchecked)) {
			throw new ScalingException(
					String.format("Scaling \"%s\" cannot scale the value \"%s\".",
							this, value));
		}
		
		return unchecked;
	}
	
	public double invert(double value) throws ScalingException {
		return inverse().scale(value);
	}
	
	public boolean isScalable(double value) {
		try {
			scale(value);
			return true;
		} catch (ScalingException e) {
			LogStream.DEBUG.send(e, "Failed to scale value %f.", value);
			return false;
		}
	}
}
