package edu.iu.sci2.visualization.geomaps.viz;

import java.util.Dictionary;
import java.util.List;

import org.cishell.utilities.mutateParameter.dropdown.DropdownMutator;

import com.google.common.base.Objects;
import com.google.common.collect.Range;

import edu.iu.sci2.visualization.geomaps.data.GeoDatum;
import edu.iu.sci2.visualization.geomaps.data.scaling.Scaling;
import edu.iu.sci2.visualization.geomaps.data.scaling.ScalingException;
import edu.iu.sci2.visualization.geomaps.geo.shapefiles.Shapefile;
import edu.iu.sci2.visualization.geomaps.viz.coding.Coding;
import edu.iu.sci2.visualization.geomaps.viz.strategy.Strategy;

/**
 * A quantitative aspect of data suitable for visualization with methods for hooking into algorithm
 * parameters.
 * 
 * <p>
 * It is recommended that implementations are enums that together give all suitable codings for a
 * particular kind of visualization.
 */
public interface VizDimension {
	String getColumnNameParameterId();
	String getColumnNameParameterDisablingToken();
	String getScalingParameterId();
	String getRangeParameterId();
	
	Strategy defaultStrategy();
	Binding<? extends VizDimension> bindingFor(Dictionary<String, Object> parameters);
	
	void addOptionsToAlgorithmParameters(DropdownMutator mutator, List<String> numericColumnNames);
	
	
	/**
	 * A binding of a {@link VizDimension} to a set of algorithm parameters as generated by
	 * {@link VizDimension#bindingFor(Dictionary)}.
	 * 
	 * <p>A VizDimension knows compile-time algorithm parameter information and its Binding knows
	 * run-time algorithm parameter information. 
	 * 
	 * @param <D>	The VizDimension being bound
	 */
	abstract class Binding<D extends Enum<D> & VizDimension> {		
		private final D dimension;
		private final String columnName;
		private final Scaling scaling;
		
		protected Binding(D dimension, Dictionary<String, Object> parameters) {
			this.dimension = dimension;
			this.columnName = (String) parameters.get(dimension.getColumnNameParameterId());
			this.scaling = Scaling.valueOf((String) parameters.get(dimension.getScalingParameterId()));
		}
		
		/**
		 * Generate a {@link Coding} for this bound VizDimension.
		 */
		public abstract Coding<D> codingForDataRange(
				Range<Double> usableRange, Range<Double> scaledRange, Shapefile shapefile);
		
		public D dimension() {
			return dimension;
		} 
		
		public String columnName() {
			return columnName;
		}
		
		public Scaling scaling() {
			return scaling;
		}
		
		public boolean isEnabled() {
			return (!Objects.equal(columnName, dimension.getColumnNameParameterDisablingToken()));
		}
		
		public <G> boolean isScalable(GeoDatum<G, D> geoDatum) {
			return scaling().isScalable(geoDatum.valueInDimension(dimension()));
		}
		
		public <G> double scale(GeoDatum<G, D> geoDatum) throws ScalingException {
			return scaling().scale(geoDatum.valueInDimension(dimension()));
		}
	}
}